/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: DEV_FileUser.c 12947 2019-04-10 14:18:28Z LuisContreras $:

  Description:
    General example functions for file handling

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2019-04-05  Use Hilscher Definitions
    2018-08-08  review
    2018-07-26  Changed memory area initialization to OS_Memset()
    2012-03-05  initial version

**************************************************************************************/

#include "cifXHWFunctions.h"
#include "cifXEndianess.h"
/* Error & Type Def. */
#include "CIFXErrors.h"

#include "DEV_FileUser.h"

#include "Hil_Results.h"
#include "Hil_SystemCmd.h"

/* ============================================================================= */
/* Global infomation and definitions                                             */
/* ============================================================================= */

/*****************************************************************************/
/*! Starts directory enumeration on the given channel
*   \param hSysdevice       Handle to the system device
*   \param ulChannel        Channel number to get directory from
*   \param ptDirectoryInfo  Pointer to enumeration result.
*                           (Will be initialized inside function)
*   \param pfnRecvPktCallback Callback for unhandled packets
*   \param pvUser             User data for callback function
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t DEV_FindFirstFile(PCHANNELINSTANCE ptChannel, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo,
                          PFN_RECV_PKT_CALLBACK       pfnRecvPktCallback, void* pvUser)
{
  int32_t             lRet         = CIFX_NO_ERROR;
  union
  {
    CIFX_PACKET        tPacket;
    HIL_DIR_LIST_REQ_T tDirListReq;

  } uSendPacket;
  HIL_DIR_LIST_CNF_T  tDirListCnf;

  (void)OS_Memset( &uSendPacket, 0, sizeof(uSendPacket));
  (void)OS_Memset( &tDirListCnf, 0, sizeof(tDirListCnf));

  if(OS_Strlen(ptDirectoryInfo->szFilename) > 0)
  {
    uint16_t usDirNameLength = (uint16_t)(OS_Strlen(ptDirectoryInfo->szFilename) + 1);

    uSendPacket.tDirListReq.tData.usDirNameLength = HOST_TO_LE16(usDirNameLength);
    (void)OS_Strncpy( (char*)((&uSendPacket.tDirListReq.tData) + 1),
                      ptDirectoryInfo->szFilename,
                      usDirNameLength);
  }

  uSendPacket.tDirListReq.tHead.ulDest      = HOST_TO_LE32(HIL_PACKET_DEST_SYSTEM);
  uSendPacket.tDirListReq.tHead.ulSrc       = HOST_TO_LE32(((PDEVICEINSTANCE)(ptChannel->pvDeviceInstance))->ulPhysicalAddress);
  uSendPacket.tDirListReq.tHead.ulCmd       = HOST_TO_LE32(HIL_DIR_LIST_REQ);
  uSendPacket.tDirListReq.tHead.ulLen       = HOST_TO_LE32( (sizeof(uSendPacket.tDirListReq.tData) +
                                                            uSendPacket.tDirListReq.tData.usDirNameLength) );
  uSendPacket.tDirListReq.tData.ulChannelNo = HOST_TO_LE32(ulChannel);


  lRet = DEV_TransferPacket(ptChannel,
                            &uSendPacket.tPacket,
                            (CIFX_PACKET*)&tDirListCnf,
                            sizeof(tDirListCnf),
                            CIFX_TO_SEND_PACKET,
                            pfnRecvPktCallback,
                            pvUser);

  if( CIFX_NO_ERROR == lRet)
  {
    if( SUCCESS_HIL_OK == (lRet = LE32_TO_HOST(tDirListCnf.tHead.ulSta)) )
    {
      /* TODO: Store handle for directory list, which needs to be set by firmware */
      ptDirectoryInfo->hList = (void*)1;
      (void)OS_Strncpy( ptDirectoryInfo->szFilename,
                        (const char*)tDirListCnf.tData.szName,
                        sizeof(ptDirectoryInfo->szFilename));

      ptDirectoryInfo->bFiletype  = tDirListCnf.tData.bFileType;
      ptDirectoryInfo->ulFilesize = LE32_TO_HOST(tDirListCnf.tData.ulFileSize);
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! Enumerate next entry in directoy on the given channel
*   \param hSysdevice       Handle to the system device
*   \param ulChannel        Channel number to get directory from
*   \param ptDirectoryInfo  Pointer to enumeration result
*   \param pfnRecvPktCallback Callback for unhandled packets
*   \param pvUser             User data for callback function
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t DEV_FindNextFile(CIFXHANDLE hSysdevice, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo,
                         PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback, void* pvUser)
{
  int32_t             lRet         = CIFX_NO_ERROR;
  PCHANNELINSTANCE    ptChannel    = (PCHANNELINSTANCE)hSysdevice;
  union
  {
    CIFX_PACKET         tPacket;
    HIL_DIR_LIST_REQ_T  tDirListReq;

  } uSendPacket;
  HIL_DIR_LIST_CNF_T  tDirListCnf;
  uint16_t            usDirNameLen = 0;

  OS_Memset( &uSendPacket, 0, sizeof(uSendPacket));
  OS_Memset( &tDirListCnf, 0, sizeof(tDirListCnf));

#ifdef CIFX_TOOLKIT_PARAMETER_CHECK
  if ( (CIFX_NO_ERROR != CheckSysdeviceHandle(hSysdevice)) &&
       (CIFX_NO_ERROR != CheckChannelHandle(hSysdevice))     )
       return CIFX_INVALID_HANDLE;
#endif

  usDirNameLen = (uint16_t)(OS_Strlen(ptDirectoryInfo->szFilename) + 1);
  uSendPacket.tDirListReq.tData.usDirNameLength = HOST_TO_LE16(usDirNameLen);
  (void)OS_Strncpy( (char*)((&uSendPacket.tDirListReq.tData) + 1),
                    ptDirectoryInfo->szFilename,
                    usDirNameLen);

  uSendPacket.tDirListReq.tHead.ulDest      = HOST_TO_LE32(HIL_PACKET_DEST_SYSTEM);
  uSendPacket.tDirListReq.tHead.ulSrc       = HOST_TO_LE32(((PDEVICEINSTANCE)(ptChannel->pvDeviceInstance))->ulPhysicalAddress);
  uSendPacket.tDirListReq.tHead.ulCmd       = HOST_TO_LE32(HIL_DIR_LIST_REQ);
  uSendPacket.tDirListReq.tHead.ulLen       = HOST_TO_LE32( (sizeof(uSendPacket.tDirListReq.tData) + usDirNameLen) );
  uSendPacket.tDirListReq.tHead.ulExt       = HOST_TO_LE32(HIL_PACKET_SEQ_MIDDLE);

  uSendPacket.tDirListReq.tData.ulChannelNo = HOST_TO_LE32(ulChannel);

  lRet = DEV_TransferPacket(ptChannel,
                            &uSendPacket.tPacket,
                            (CIFX_PACKET*)&tDirListCnf,
                            sizeof(tDirListCnf),
                            CIFX_TO_SEND_PACKET,
                            pfnRecvPktCallback,
                            pvUser);

  if(CIFX_NO_ERROR == lRet)
  {
    if( SUCCESS_HIL_OK == (lRet = (LE32_TO_HOST(tDirListCnf.tHead.ulSta))) )
    {
      if(( LE32_TO_HOST(tDirListCnf.tHead.ulExt) & HIL_PACKET_SEQ_MASK) == HIL_PACKET_SEQ_LAST)
      {
        /* this is the last packet */
        lRet = CIFX_NO_MORE_ENTRIES;

        /* invalidate handle */
        ptDirectoryInfo->hList = (void*)0;

      } else
      {
        (void)OS_Strncpy( ptDirectoryInfo->szFilename,
                          (const char*)tDirListCnf.tData.szName,
                          sizeof(tDirListCnf.tData.szName));

        ptDirectoryInfo->bFiletype  = tDirListCnf.tData.bFileType;
        ptDirectoryInfo->ulFilesize = LE32_TO_HOST(tDirListCnf.tData.ulFileSize);
      }
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! Delete the given file
*   \param pvChannel          Channel instance
*   \param ulChannelNumber    Channel number
*   \param pszFileName        Input file name
*   \param pfnTransferPacket  Function used for transferring packets
*   \param pfnRecvPacket      User callback for unsolicited receive packets
*   \param pvUser             User parameter passed on callback
*   \return 1 on success                                                     */
/*****************************************************************************/
int32_t DEV_DeleteFile(void* pvChannel, uint32_t ulChannelNumber, char* pszFileName,
                       PFN_TRANSFER_PACKET    pfnTransferPacket,
                       PFN_RECV_PKT_CALLBACK  pfnRecvPacket,
                       void*                  pvUser)
{
  /* Create delete packet */
  union
  {
    CIFX_PACKET           tPacket;
    HIL_FILE_DELETE_REQ_T tFileDelete;

  } uSendPkt;
  CIFX_PACKET tConf;
  uint16_t    usFileNameLen = (uint16_t)OS_Strlen(pszFileName);
  int32_t     lRet          = CIFX_NO_ERROR;

  uint32_t    ulSrc         = OS_GetMilliSecCounter(); /* Early versions used pvChannel as ulSrc,
                                                          but this won't work on 64 Bit machines.
                                                          As we need something unique we use the current system time */

  (void)OS_Memset( &uSendPkt, 0, sizeof(uSendPkt));
  (void)OS_Memset( &tConf, 0, sizeof(tConf));

  /* Initialize the message */
  uSendPkt.tFileDelete.tHead.ulSrc    = HOST_TO_LE32(ulSrc);
  uSendPkt.tFileDelete.tHead.ulDest   = HOST_TO_LE32(HIL_PACKET_DEST_SYSTEM);
  uSendPkt.tFileDelete.tHead.ulCmd    = HOST_TO_LE32(HIL_FILE_DELETE_REQ);
  uSendPkt.tFileDelete.tHead.ulExt    = HOST_TO_LE32(HIL_PACKET_SEQ_NONE);
  uSendPkt.tFileDelete.tHead.ulLen    = HOST_TO_LE32((uint32_t)(sizeof(uSendPkt.tFileDelete.tData) +
                                                                usFileNameLen + 1));

  /* Insert file data */
  uSendPkt.tFileDelete.tData.ulChannelNo      = HOST_TO_LE32(ulChannelNumber);
  uSendPkt.tFileDelete.tData.usFileNameLength = HOST_TO_LE16( (usFileNameLen + 1) );

  /* Insert file name */
  (void)OS_Strncpy( (char*)(&uSendPkt.tFileDelete.tData + 1),
                     pszFileName,
                    (uint32_t)(sizeof(uSendPkt.tPacket.abData) - sizeof(uSendPkt.tFileDelete.tData)));

  /* Send delete packet */
  lRet = pfnTransferPacket( pvChannel,
                            &uSendPkt.tPacket,
                            (CIFX_PACKET*)&tConf,
                            (uint32_t)sizeof(tConf),
                            CIFX_TO_FIRMWARE_START,       /* Could take a little while */
                            pfnRecvPacket,
                            pvUser);

  if(CIFX_NO_ERROR == lRet)
    lRet = LE32_TO_HOST(tConf.tHeader.ulState);

  return lRet;
}

/*****************************************************************************/
/*! Delete all existing files in a channel, from the file system.
*   \param ptChannel          Channel instance
*   \param ulChannel          Channel number
*   \param pfnTransferPacket  Function used for transferring packets
*   \param pfnRecvPacket      User callback for unsolicited receive packets
*   \param pvUser             User parameter passed on callback
*   \param szExceptFile       File extension to ignore while deleting files
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t DEV_RemoveChannelFiles(PCHANNELINSTANCE ptChannel, uint32_t ulChannel,
                               PFN_TRANSFER_PACKET    pfnTransferPacket,
                               PFN_RECV_PKT_CALLBACK  pfnRecvPacket,
                               void*                  pvUser,
                               char*                  szExceptFile)
{
  /* Try to find file with the extension *.nxm, *.nxf, *.mod and remove it */
  CIFX_DIRECTORYENTRY tDirectoryEntry;
  int32_t             lRet            = CIFX_NO_ERROR;
  int                 fFindFirst      = 1;

  /* Search for all firmware files. If one is found. delete it an start with find first again, */
  /* because we can't store a directory list in here */
  do
  {
    if ( fFindFirst)
    {
      OS_Memset( &tDirectoryEntry, 0, sizeof(tDirectoryEntry));

      /* Search first file */
      if ( !(CIFX_NO_ERROR == (lRet = DEV_FindFirstFile( ptChannel, ulChannel, &tDirectoryEntry, pfnRecvPacket, pvUser))))
      {
        /* No more files, or error during find first */
        break;
      } else
      {
        /* Is this a valid file name */
        int iStrlen = OS_Strlen(tDirectoryEntry.szFilename);
        if( iStrlen >= CIFX_MIN_FILE_NAME_LENGTH)  /* At least x.abc */
        {
          if( !((NULL != szExceptFile)                                                          &&
                (4 == OS_Strlen(szExceptFile))                                                  &&
                (0 == OS_Strnicmp( szExceptFile, &tDirectoryEntry.szFilename[iStrlen - 4], 4)))   )
          {
            /* Delete file and continue with find first file again */
            (void)DEV_DeleteFile( ptChannel, ulChannel, tDirectoryEntry.szFilename, pfnTransferPacket, pfnRecvPacket, pvUser);
          }
        } else
        {
          /* Not a valid file, search next file */
          fFindFirst = 0;
        }
      }
    } else
    {
      /* Search for more files */
      if ( !(CIFX_NO_ERROR == (lRet = DEV_FindNextFile( ptChannel, ulChannel, &tDirectoryEntry, pfnRecvPacket, pvUser))))
      {
        /* No more files, or error during find next */
        break;
      } else
      {
        /* Is this a valid file name */
        int iStrlen = OS_Strlen(tDirectoryEntry.szFilename);
        if( iStrlen >= CIFX_MIN_FILE_NAME_LENGTH)  /* At least x.abc */
        {
          /* If firmware file, delete it, else search until all files checked */
          if( !((NULL != szExceptFile)                                            &&
                (4 == OS_Strlen(szExceptFile))                                    &&
                (0 == OS_Strnicmp( szExceptFile, &tDirectoryEntry.szFilename[iStrlen - 4], 4)))   )
          {
            /* Delete the file and start with find first again */
            (void)DEV_DeleteFile( ptChannel, ulChannel, tDirectoryEntry.szFilename, pfnTransferPacket, pfnRecvPacket, pvUser);
            fFindFirst = 1;
          }
        }
      }
    }
  } while ( CIFX_NO_ERROR == lRet);

  return 1;
}
